//****************************************************************************
// 
// boolean_array.js
// Scott Brause
// 2003-02-28
//
// Contains the BooleanArray Object constructor and member functions.
//****************************************************************************

//****************************************************************************
//
// BooleanArray()
// The constructor for the BooleanArray Object.
//
// The BooleanArray is an Object used to manage a 0-indexed array of
// true-false data items. In addition to just storing the data items,
// the BooleanArray Object can also export the information in an
// ASCII encoded String, as well as import information from a previously
// exported ASCII encoded String.
//
// Input Params:     
// strVal (Optional) - String containing information to be imported.  If not
//                     supplied, the BooleanArray will be initialized with
//                     an empty array.
//
// Output Params:
//
// Returns:
// A BooleanArray Object
//
// Calls/References:
// BooleanArrayExport - acts as a member function for exporting data.
// BooleanArrayImport - acts as a member function for loading data.
//
//****************************************************************************
//
// Member Functions:
// Boolean set(Integer intIndex, Boolean bolVal)
//   Sets the value for the array item at index intIndex
//   to the bolVal. Returns bolVal.
// Boolean get(Integer intIndex)
//   Returns the value stored in the array at index intIndex.
// Boolean load(String strLoadString)
//   Clears Object's data and loads in new data values
//   from the encoded String, strLoadString.  Returns
//   true if no errors were encountered during loading;
//   returns false otherwise.
// String dump()
//   Returns an encoded String containing all of the information
//   stored in the Object, in an compressed form (up to 6 boolean
//   values are stored in each character of the returned String).
//
//****************************************************************************
function BooleanArray(strVal)
{
	// Array used to store all true-false data
	this.arr = new Array();
	
	// If a parameter was passed to the constructor
	// then import that data into the Object.
	if (strVal != null)
	{
		this.load(strVal);
	}
	
	// Return a pointer to the Object.
	return this;
}
// Member function definitions
BooleanArray.prototype.set = function (intIndex, bolVal) {this.arr[intIndex] = bolVal;return bolVal};
BooleanArray.prototype.get = function (intIndex) {return this.arr[intIndex]};
BooleanArray.prototype.load = BooleanArrayImport;
BooleanArray.prototype.dump = BooleanArrayExport;
BooleanArray.prototype.getLength = BooleanArrayGetLength;

//****************************************************************************
//
// BooleanArrayGetLength()
// Member function definition for the BooleanArray Object's getLength() function.
//
// This function will return the length of the BooleanArray (the highest index
// set, plus one).
//
// Input Params:     
//
// Output Params:
//
// Returns:
// An integer representing the length of the BooleanArray.
//
// Calls/References:
//
//****************************************************************************
function BooleanArrayGetLength()
{
	return this.arr.length;
}
	
//****************************************************************************
//
// BooleanArrayExport()
// Member function definition for the BooleanArray Object's dump() function.
//
// This function will return a String that contains all of the true-false
// data from the object's arr[] array.  The true-false information is encoded
// into individual bits (0->false, 1->true), and then assembled as a stream of
// character bytes. Although, technically, an 8-bit byte can represent 256
// distinct values, we cannot use all of the bits.
//
// We cannot use the low-order bit because a byte with the value 0 [00000000]
// acts to terminate the string, effectively wiping out all data that may 
// come after it. The solution we employ is to always set the low-order bit
// to 1 - leaving us with only 7 bits to work with in each byte.
//
// But there is also the problem that not all processes that may have to
// use/store/manipulate the output String will support byte-values that 
// extend beyond the ASCII 7-bit encoding. This means that we can only output 
// byte-values that are between 0 and 127. So, in addition to nullifying the
// low-order bit, we will also need to nullify the high-order bit as well by
// always setting the 8th bit to 0 - leaving us with the middle 6 bits of the
// byte to store the true-false data in.
//
// Therefore the compression achieved is about 6:1 (6 true-false data elements
// are stored in one encoded character byte).
//
// Input Params:     
//
// Output Params:
//
// Returns:
// An encoded String representation of all of the true-false information
// contained in the object's arr[] array.
//
// Calls/References:
//
//****************************************************************************
function BooleanArrayExport()
{
	// The string always begins with a character representation
	// of the number of elements in the array, and the "*" character
	// which acts as a delimiter.
	var strVal = this.arr.length + "*";
	var intPosition = 1;
	var intAccumulator = 1;
	
	// Step through all elements of the arr[] array, translating
	// each true into a bit value of 1, and each false into a bit
	// value of 0.  As the true-false values are converted into 
	// bit values, they are shifted left and "AND-ed" into an 
	// accumulator. When the accumulator is full (6-bits have
	// been inserted), the accumulator byte is added to the output
	// String, the accumulator is reset, and processing continues
	// until the entire arr[] array has been processed.
	for (var i=0;i<this.arr.length;i++)
	{
		intAccumulator = intAccumulator | ( ((this.arr[i] == true)?(1):(0)) << intPosition);
		if (intPosition == 6 || i == this.arr.length-1)
		{
			strVal += String.fromCharCode(intAccumulator);
			intPosition = 1;
			intAccumulator = 1;
		}
		else
		{
			intPosition++;
		}
	}
	
	return strVal;
}

//****************************************************************************
//
// BooleanArrayImport()
// Member function definition for the BooleanArray object's load() function.
//
// Input Params:     
//
// Output Params:
//
// Returns:
// Returns an encoded String representation of all of the true-false information
// contained in the object's arr[] array.
//
// Calls/References:
//
//****************************************************************************
function BooleanArrayImport(strVal)
{
	// intIndex is used to determine where the
	// first "*" character resides (the "*" character
	// marks where the "length" portion of the String
	// ends, and the encoded byte-stream begins).
	var intIndex = 0;
	// intNumBits will be used to hold the number of
	// meaningful bits in the stream.
	var intNumBits;
	
	// Reset the arr[] array.
	this.arr = new Array();
	
	// Loop through the String, looking for
	// the first ocurrence of the "*" character.
	for (var i=0;i<strVal.length;i++)
	{
		if (strVal.charAt(i) == "*")
		{
			// The "*" character was found, so set
			// intIndex and break out of the loop;
			intIndex = i;
			break;
		}
	}
	
	// Get length information from the leading
	// intIndex-1 number of characters of the String.
	intNumBits = parseInt(strVal.slice(0,intIndex));
	// If the leading value doesn't parse-out as an
	// integer, or it parses-out as a negative number,
	// then strVal is invalid, and the function will
	// be aborted.
	if ( (typeof(intNumBits) != "number") || (intNumBits < 0) )
	{
		return false;
	}

	// Make intNumBits to hold the number
	// of "left-over" bits held in the
	// last byte of strVal.
	intNumBits = intNumBits % 6;
	
	// Loop through strVal, converting
	// bits into true-false entries
	// and placing them in the arr[] array.
	for (var i=intIndex+1;i<strVal.length;i++)
	{
		var lastBit = 6;
		if (i == strVal.length-1 && intNumBits > 0)
		{
			lastBit = intNumBits;
		}
		for (var t=1;t<=lastBit;t++)
		{
			this.arr[this.arr.length] = (((strVal.charCodeAt(i) & Math.pow(2,t)) > 0)?(true):(false));
		}
	}
}

